/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.picketlink.social.openid.providers.helpers;

import javax.xml.namespace.QName;

import org.picketlink.identity.federation.core.interfaces.ProtocolContext;
import org.picketlink.identity.federation.core.interfaces.SecurityTokenProvider;
import org.picketlink.social.openid.providers.OpenIDTokenProvider;
import org.picketlink.social.openid.providers.helpers.OpenIDProviderManager.OpenIDMessage;

/**
 * {@code ProtocolContext} for the OpenID Layer
 *
 * @author Anil.Saldhana@redhat.com
 * @since Jan 20, 2011
 */
public class OpenIDProtocolContext implements ProtocolContext {
    /**
     * The response message generated by the STS
     */
    protected OpenIDMessage responseMessage;

    /**
     * The list of request parameters for use by the STS
     */
    protected OpenIDParameterList requestParameterList;

    /**
     * The Server Manager needs an endpoint
     */
    protected String endpoint;

    /**
     * Should the STS issue an error?
     */
    protected Boolean issueError = Boolean.FALSE;

    /**
     * What is the text the STS should use for the error?
     */
    protected String errorText = null;

    /**
     * The OpenID mode
     */
    protected MODE mode;

    /**
     * STS uses the AUTH_HOLDER information for processing authentication requests
     */
    protected AUTH_HOLDER authenticationHolder;

    /**
     * An Enum indicating the OpenID mode
     */
    public enum MODE {
        ASSOCIATE, CHECK_ID_SETUP, CHECK_ID_IMMEDIATE, CHECK_AUTHENTICATION
    };

    /**
     * Class to hold the information for authentication requests
     */
    public static class AUTH_HOLDER {
        private String userSelectedId, userSelectedClaimedId;
        private boolean authenticatedAndApproved;

        public AUTH_HOLDER(String userSelectedId, String userSelectedClaimedId, boolean authenticatedAndApproved) {
            this.userSelectedId = userSelectedId;
            this.userSelectedClaimedId = userSelectedClaimedId;
            this.authenticatedAndApproved = authenticatedAndApproved;
        }

        @Override
        public String toString() {
            return "AUTH_HOLDER [userSelectedId=" + userSelectedId + ", userSelectedClaimedId=" + userSelectedClaimedId
                    + ", authenticatedAndApproved=" + authenticatedAndApproved + "]";
        }

        public String getUserSelectedId() {
            return userSelectedId;
        }

        public String getUserSelectedClaimedId() {
            return userSelectedClaimedId;
        }

        public boolean isAuthenticatedAndApproved() {
            return authenticatedAndApproved;
        }
    }

    /**
     * Get the holder of authentication information
     *
     * @return
     */
    public AUTH_HOLDER getAuthenticationHolder() {
        return authenticationHolder;
    }

    /**
     * Set an authentication holder
     *
     * @param authenticationHolder
     */
    public void setAuthenticationHolder(AUTH_HOLDER authenticationHolder) {
        this.authenticationHolder = authenticationHolder;
    }

    /**
     * Get the Open ID mode
     *
     * @return
     */
    public MODE getMode() {
        return this.mode;
    }

    /**
     * Set the OpenID mode
     *
     * @param theMode
     */
    public void setMode(MODE theMode) {
        this.mode = theMode;
    }

    /**
     * Get the error text (Null by default)
     *
     * @return
     */
    public String getErrorText() {
        return errorText;
    }

    /**
     * Set the error text
     *
     * @param errorText
     */
    public void setErrorText(String errorText) {
        this.errorText = errorText;
    }

    /**
     * Should the STS issue an error response token
     *
     * @return
     */
    public Boolean getIssueError() {
        return issueError;
    }

    /**
     * Set the error response token for the STS
     *
     * @param issueError
     */
    public void setIssueError(Boolean issueError) {
        this.issueError = issueError;
    }

    /**
     * Get the endpoint address
     *
     * @return
     */
    public String getEndpoint() {
        return endpoint;
    }

    /**
     * Set the endpoint address
     *
     * @param endpoint
     */
    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    /**
     * Get the request parameter list
     *
     * @return an instanceof {@code OpenIDParameterList}
     */
    public OpenIDParameterList getRequestParameterList() {
        return requestParameterList;
    }

    /**
     * Set the request parameter list
     *
     * @param requestParameterList an instanceof {@code OpenIDParameterList}
     */
    public void setRequestParameterList(OpenIDParameterList requestParameterList) {
        this.requestParameterList = requestParameterList;
    }

    /**
     * Get the {@code OpenIDMessage} response message that the STS has generated
     *
     * @return
     */
    public OpenIDMessage getResponseMessage() {
        return responseMessage;
    }

    /**
     * Set the {@code OpenIDMessage} response message
     *
     * @param responseMessage
     */
    public void setResponseMessage(OpenIDMessage responseMessage) {
        this.responseMessage = responseMessage;
    }

    /**
     * @see org.picketlink.identity.federation.core.interfaces.ProtocolContext#serviceName()
     */
    public String serviceName() {
        return null;
    }

    /**
     * @see org.picketlink.identity.federation.core.interfaces.ProtocolContext#tokenType()
     */
    public String tokenType() {
        return OpenIDTokenProvider.OPENID_1_0_NS;
    }

    /**
     * @see org.picketlink.identity.federation.core.interfaces.ProtocolContext#getQName()
     */
    public QName getQName() {
        return new QName(OpenIDTokenProvider.OPENID_1_0_NS);
    }

    /**
     * @see org.picketlink.identity.federation.core.interfaces.ProtocolContext#family()
     */
    public String family() {
        return SecurityTokenProvider.FAMILY_TYPE.OPENID.name();
    }
}