/*
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package elemental2.dom;

import elemental2.core.ArrayBuffer;
import elemental2.core.ArrayBufferView;
import java.lang.Object;
import java.lang.String;
import jsinterop.annotations.JsFunction;
import jsinterop.annotations.JsOverlay;
import jsinterop.annotations.JsPackage;
import jsinterop.annotations.JsProperty;
import jsinterop.annotations.JsType;
import jsinterop.base.Js;

@JsType(isNative = true, namespace = JsPackage.GLOBAL)
public interface RTCDataChannel extends EventTarget {
  @JsFunction
  public interface OncloseCallbackFn {
    Object onInvoke(Event p0);
  }

  @JsFunction
  public interface OnerrorCallbackFn {
    Object onInvoke(Event p0);
  }

  @JsFunction
  public interface OnmessageCallbackFn {
    Object onInvoke(MessageEvent p0);
  }

  @JsFunction
  public interface OnopenCallbackFn {
    Object onInvoke(Event p0);
  }

  @JsType(isNative = true, name = "?", namespace = JsPackage.GLOBAL)
  public interface SendDataUnionType {
    @JsOverlay
    static SendDataUnionType of(Object o) {
      return Js.cast(o);
    }

    @JsOverlay
    default ArrayBuffer asArrayBuffer() {
      return Js.cast(this);
    }

    @JsOverlay
    default ArrayBufferView asArrayBufferView() {
      return Js.cast(this);
    }

    @JsOverlay
    default Blob asBlob() {
      return Js.cast(this);
    }

    @JsOverlay
    default String asString() {
      return Js.cast(this);
    }

    @JsOverlay
    default boolean isArrayBuffer() {
      return (Object) this instanceof ArrayBuffer;
    }

    @JsOverlay
    default boolean isArrayBufferView() {
      return (Object) this instanceof ArrayBufferView;
    }

    @JsOverlay
    default boolean isBlob() {
      return (Object) this instanceof Blob;
    }

    @JsOverlay
    default boolean isString() {
      return (Object) this instanceof String;
    }
  }

  Object close();

  @JsProperty
  String getBinaryType();

  @JsProperty
  double getBufferedAmount();

  @JsProperty
  String getLabel();

  @JsProperty
  OncloseCallbackFn getOnclose();

  @JsProperty
  OnerrorCallbackFn getOnerror();

  @JsProperty
  OnmessageCallbackFn getOnmessage();

  @JsProperty
  OnopenCallbackFn getOnopen();

  @JsProperty
  String getReadyState();

  @JsProperty
  boolean isReliable();

  @JsOverlay
  default void send(ArrayBuffer data) {
    send(Js.<SendDataUnionType>uncheckedCast(data));
  }

  @JsOverlay
  default void send(ArrayBufferView data) {
    send(Js.<SendDataUnionType>uncheckedCast(data));
  }

  @JsOverlay
  default void send(Blob data) {
    send(Js.<SendDataUnionType>uncheckedCast(data));
  }

  void send(SendDataUnionType data);

  @JsOverlay
  default void send(String data) {
    send(Js.<SendDataUnionType>uncheckedCast(data));
  }

  @JsProperty
  void setBinaryType(String binaryType);

  @JsProperty
  void setBufferedAmount(double bufferedAmount);

  @JsProperty
  void setOnclose(OncloseCallbackFn onclose);

  @JsProperty
  void setOnerror(OnerrorCallbackFn onerror);

  @JsProperty
  void setOnmessage(OnmessageCallbackFn onmessage);

  @JsProperty
  void setOnopen(OnopenCallbackFn onopen);

  @JsProperty
  void setReadyState(String readyState);
}
