/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.as.repository;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Stream;
import org.jboss.as.repository.ContentFilter;
import org.jboss.as.repository.ContentReference;
import org.jboss.as.repository.ContentRepository;
import org.jboss.as.repository.ContentRepositoryElement;
import org.jboss.as.repository.ExplodedContent;
import org.jboss.as.repository.ExplodedContentException;
import org.jboss.as.repository.HashUtil;
import org.jboss.as.repository.PathUtil;
import org.jboss.as.repository.TemporaryFileInputStream;
import org.jboss.as.repository.TypedInputStream;
import org.jboss.as.repository.logging.DeploymentRepositoryLogger;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;
import org.wildfly.common.Assert;

public class ContentRepositoryImpl
implements ContentRepository {
    protected static final String CONTENT = "content";
    private final File repoRoot;
    private final File tmpRoot;
    protected final AtomicReference<MessageDigest> messageDigestRef;
    private final Map<String, Set<ContentReference>> contentHashReferences = new HashMap<String, Set<ContentReference>>();
    private final Map<String, ReentrantLock> lockedContents = new HashMap<String, ReentrantLock>();
    private final Map<String, Long> obsoleteContents = new HashMap<String, Long>();
    private final long obsolescenceTimeout;
    private final long lockTimeout;
    private volatile boolean readWrite = false;

    protected ContentRepositoryImpl(File repoRoot, File tmpRoot, long obsolescenceTimeout, long lockTimeout) {
        Assert.checkNotNullParam((String)"repoRoot", (Object)repoRoot);
        Assert.checkNotNullParam((String)"tmpRoot", (Object)tmpRoot);
        this.checkDirectory(repoRoot);
        this.repoRoot = repoRoot;
        this.checkDirectory(tmpRoot);
        this.tmpRoot = tmpRoot;
        this.obsolescenceTimeout = obsolescenceTimeout;
        this.lockTimeout = lockTimeout;
        this.messageDigestRef = new AtomicReference<MessageDigest>(ContentRepositoryImpl.createMessageDigest());
    }

    private void checkDirectory(File directory) {
        if (directory.exists()) {
            if (!directory.isDirectory()) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.notADirectory(directory.getAbsolutePath());
            }
            if (!directory.canWrite()) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.directoryNotWritable(directory.getAbsolutePath());
            }
        } else if (!directory.mkdirs()) {
            throw DeploymentRepositoryLogger.ROOT_LOGGER.cannotCreateDirectory(null, directory.getAbsolutePath());
        }
    }

    @Override
    public void readWrite() {
        this.readWrite = true;
    }

    @Override
    public void readOnly() {
        this.readWrite = false;
    }

    @Override
    public byte[] addContent(InputStream stream) throws IOException {
        byte[] sha1Bytes;
        Path tmp;
        block17: {
            tmp = File.createTempFile(CONTENT, ".tmp", this.repoRoot).toPath();
            if (stream != null) {
                try (OutputStream fos = Files.newOutputStream(tmp, new OpenOption[0]);
                     MessageDigestHandle digestHandle = new MessageDigestHandle();){
                    int read;
                    MessageDigest messageDigest = digestHandle.getMessageDigest();
                    DigestOutputStream dos = new DigestOutputStream(fos, messageDigest);
                    BufferedInputStream bis = new BufferedInputStream(stream);
                    byte[] bytes = new byte[8192];
                    while ((read = bis.read(bytes)) > -1) {
                        dos.write(bytes, 0, read);
                    }
                    fos.flush();
                    sha1Bytes = messageDigest.digest();
                    break block17;
                }
            }
            Files.delete(tmp);
            Files.createDirectory(tmp, new FileAttribute[0]);
            sha1Bytes = this.getSha1Bytes(tmp);
        }
        Path realFile = this.getDeploymentContentFile(sha1Bytes, true);
        if (this.hasContent(sha1Bytes)) {
            try {
                PathUtil.deleteRecursively(tmp);
            }
            catch (IOException ioex) {
                DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(ioex, tmp.toAbsolutePath().toString());
                tmp.toFile().deleteOnExit();
            }
            DeploymentRepositoryLogger.ROOT_LOGGER.debugf("Content was already present in repository at location %s", realFile.toAbsolutePath().toString());
        } else {
            this.moveTempToPermanent(tmp, realFile);
            DeploymentRepositoryLogger.ROOT_LOGGER.contentAdded(realFile.toAbsolutePath().toString());
        }
        return sha1Bytes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void addContentReference(ContentReference reference) {
        if (!this.readWrite) {
            return;
        }
        Map<String, Set<ContentReference>> map = this.contentHashReferences;
        synchronized (map) {
            Set references = this.contentHashReferences.computeIfAbsent(reference.getHexHash(), k -> new HashSet());
            references.add(reference);
        }
    }

    @Override
    public VirtualFile getContent(byte[] hash) {
        Assert.checkNotNullParam((String)"hash", (Object)hash);
        return VFS.getChild((URI)this.getDeploymentContentFile(hash, true).toUri());
    }

    @Override
    public boolean syncContent(ContentReference reference) {
        return this.hasContent(reference.getHash());
    }

    @Override
    public boolean hasContent(byte[] hash) {
        return Files.exists(this.getDeploymentContentFile(hash), new LinkOption[0]);
    }

    protected Path getRepoRoot() {
        return this.repoRoot.toPath();
    }

    protected Path getDeploymentContentFile(byte[] deploymentHash) {
        return this.getDeploymentContentFile(deploymentHash, false);
    }

    protected Path getDeploymentContentFile(byte[] deploymentHash, boolean validate) {
        return this.getDeploymentHashDir(deploymentHash, validate).resolve(CONTENT);
    }

    protected Path getDeploymentHashDir(byte[] deploymentHash, boolean validate) {
        String sha1 = HashUtil.bytesToHexString(deploymentHash);
        String partA = sha1.substring(0, 2);
        String partB = sha1.substring(2);
        Path base = this.getRepoRoot().resolve(partA);
        if (validate) {
            this.validateDir(base);
        }
        Path hashDir = base.resolve(partB);
        if (validate && !Files.exists(hashDir, new LinkOption[0])) {
            try {
                Files.createDirectories(hashDir, new FileAttribute[0]);
            }
            catch (IOException ioex) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.cannotCreateDirectory(ioex, hashDir.toAbsolutePath().toString());
            }
        }
        return hashDir;
    }

    protected void validateDir(Path dir) {
        if (!Files.exists(dir, new LinkOption[0])) {
            try {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            catch (IOException ioex) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.cannotCreateDirectory(ioex, dir.toAbsolutePath().toString());
            }
        } else {
            if (!Files.isDirectory(dir, new LinkOption[0])) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.notADirectory(dir.toAbsolutePath().toString());
            }
            if (!dir.toFile().canWrite()) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.directoryNotWritable(dir.toAbsolutePath().toString());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void moveTempToPermanent(Path tmpFile, Path permanentFile) throws IOException {
        Path localTmp = permanentFile.resolveSibling("tmp");
        try {
            Files.move(tmpFile, permanentFile, new CopyOption[0]);
        }
        catch (IOException ioex) {
            PathUtil.copyRecursively(tmpFile, localTmp, true);
            try {
                Files.move(localTmp, permanentFile, new CopyOption[0]);
            }
            catch (IOException ex) {
                try {
                    PathUtil.copyRecursively(localTmp, permanentFile, true);
                }
                catch (IOException e) {
                    PathUtil.deleteRecursively(permanentFile);
                    throw e;
                }
            }
        }
        finally {
            try {
                PathUtil.deleteRecursively(tmpFile);
            }
            catch (IOException ioex) {
                DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(ioex, tmpFile.toString());
                tmpFile.toFile().deleteOnExit();
            }
            try {
                PathUtil.deleteRecursively(localTmp);
            }
            catch (IOException ioex) {
                DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(ioex, localTmp.toString());
                localTmp.toFile().deleteOnExit();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void removeContent(ContentReference reference) {
        if (!this.readWrite) {
            return;
        }
        Map<String, Set<ContentReference>> map = this.contentHashReferences;
        synchronized (map) {
            Set<ContentReference> references = this.contentHashReferences.get(reference.getHexHash());
            if (references != null) {
                references.remove(reference);
                if (!references.isEmpty()) {
                    return;
                }
                this.contentHashReferences.remove(reference.getHexHash());
            }
        }
        Path contentPath = !HashUtil.isEachHexHashInTable(reference.getHexHash()) ? Paths.get(reference.getContentIdentifier(), new String[0]) : this.getDeploymentContentFile(reference.getHash(), false);
        Path parent = contentPath.getParent();
        boolean interrupted = false;
        try {
            if (HashUtil.isEachHexHashInTable(reference.getHexHash()) && this.readWrite && !this.lock(reference.getHash())) {
                DeploymentRepositoryLogger.ROOT_LOGGER.contentDeletionError(DeploymentRepositoryLogger.ROOT_LOGGER.errorLockingDeployment(), contentPath.toString());
                return;
            }
            PathUtil.deleteRecursively(parent);
        }
        catch (IOException ex) {
            DeploymentRepositoryLogger.ROOT_LOGGER.contentDeletionError(ex, contentPath.toString());
        }
        catch (InterruptedException ex) {
            interrupted = true;
            DeploymentRepositoryLogger.ROOT_LOGGER.contentDeletionError(ex, contentPath.toString());
        }
        finally {
            if (HashUtil.isEachHexHashInTable(reference.getHexHash())) {
                this.unlock(reference.getHash());
            }
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
        Path grandParent = parent.getParent();
        if (Files.exists(grandParent, new LinkOption[0])) {
            try (Stream<Path> files = Files.list(grandParent);){
                if (files.noneMatch(x$0 -> Files.isDirectory(x$0, new LinkOption[0]))) {
                    PathUtil.deleteRecursively(grandParent);
                }
            }
            catch (IOException ex) {
                DeploymentRepositoryLogger.ROOT_LOGGER.contentDeletionError(ex, grandParent.toString());
            }
        }
        DeploymentRepositoryLogger.ROOT_LOGGER.contentRemoved(contentPath.toAbsolutePath().toString());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Map<String, Set<String>> cleanObsoleteContent() {
        if (!this.readWrite) {
            return Collections.emptyMap();
        }
        HashMap<String, Set<String>> cleanedContents = new HashMap<String, Set<String>>(2);
        cleanedContents.put("marked-contents", new HashSet());
        cleanedContents.put("deleted-contents", new HashSet());
        Map<String, Set<ContentReference>> map = this.contentHashReferences;
        synchronized (map) {
            DeploymentRepositoryLogger.ROOT_LOGGER.debug("Current content hash references are " + this.contentHashReferences);
            for (ContentReference fsContent : this.listLocalContents()) {
                if (!this.readWrite) {
                    return Collections.emptyMap();
                }
                if (!this.contentHashReferences.containsKey(fsContent.getHexHash())) {
                    if (this.markAsObsolete(fsContent)) {
                        ((Set)cleanedContents.get("deleted-contents")).add(fsContent.getContentIdentifier());
                        continue;
                    }
                    ((Set)cleanedContents.get("marked-contents")).add(fsContent.getContentIdentifier());
                    continue;
                }
                this.obsoleteContents.remove(fsContent.getHexHash());
            }
        }
        return cleanedContents;
    }

    private boolean markAsObsolete(ContentReference ref) {
        if (this.obsoleteContents.containsKey(ref.getHexHash())) {
            if (this.obsoleteContents.get(ref.getHexHash()) + this.obsolescenceTimeout < System.currentTimeMillis()) {
                DeploymentRepositoryLogger.ROOT_LOGGER.obsoleteContentCleaned(ref.getContentIdentifier());
                this.removeContent(ref);
                return true;
            }
        } else {
            this.obsoleteContents.put(ref.getHexHash(), System.currentTimeMillis());
        }
        return false;
    }

    private Set<ContentReference> listLocalContents() {
        HashSet<ContentReference> localReferences = new HashSet<ContentReference>();
        File[] rootHashes = this.repoRoot.listFiles();
        if (rootHashes != null) {
            for (File rootHash : rootHashes) {
                if (!rootHash.isDirectory()) continue;
                File[] complementaryHashes = rootHash.listFiles();
                if (complementaryHashes == null || complementaryHashes.length == 0) {
                    ContentReference reference = new ContentReference(rootHash.getAbsolutePath(), rootHash.getName());
                    localReferences.add(reference);
                    continue;
                }
                for (File complementaryHash : complementaryHashes) {
                    String hash = rootHash.getName() + complementaryHash.getName();
                    ContentReference reference = new ContentReference(complementaryHash.getAbsolutePath(), hash);
                    localReferences.add(reference);
                }
            }
        } else {
            DeploymentRepositoryLogger.ROOT_LOGGER.localContentListError(this.repoRoot.getAbsolutePath());
        }
        return localReferences;
    }

    @Override
    public byte[] explodeContent(byte[] deploymentHash) throws ExplodedContentException {
        Path contentPath = this.getDeploymentContentFile(deploymentHash);
        if (!Files.exists(contentPath, new LinkOption[0])) {
            throw DeploymentRepositoryLogger.ROOT_LOGGER.archiveNotFound(contentPath.toString());
        }
        try {
            if (Files.isDirectory(contentPath, new LinkOption[0]) || !PathUtil.isArchive(contentPath)) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.notAnArchive(contentPath.toString());
            }
            Path tmp = PathUtil.createTempDirectory(this.repoRoot.toPath(), CONTENT);
            Path contentDir = Files.createDirectory(tmp.resolve(CONTENT), new FileAttribute[0]);
            PathUtil.unzip(contentPath, contentDir);
            byte[] sha1Bytes = this.getSha1Bytes(contentDir);
            Path realFile = this.getDeploymentContentFile(sha1Bytes, true);
            if (this.hasContent(sha1Bytes)) {
                try {
                    PathUtil.deleteRecursively(tmp);
                }
                catch (IOException ioex) {
                    DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(ioex, tmp.toAbsolutePath().toString());
                    tmp.toFile().deleteOnExit();
                }
                DeploymentRepositoryLogger.ROOT_LOGGER.debugf("Content was already present in repository at location %s", realFile.toAbsolutePath().toString());
            } else {
                this.moveTempToPermanent(contentDir, realFile);
                PathUtil.deleteRecursively(tmp);
                DeploymentRepositoryLogger.ROOT_LOGGER.contentExploded(realFile.toAbsolutePath().toString());
            }
            return sha1Bytes;
        }
        catch (IOException ioex) {
            DeploymentRepositoryLogger.ROOT_LOGGER.warn(ioex);
            throw DeploymentRepositoryLogger.ROOT_LOGGER.errorExplodingContent(ioex, contentPath.toString());
        }
    }

    @Override
    public byte[] explodeSubContent(byte[] deploymentHash, String relativePath) throws ExplodedContentException {
        Path contentPath = this.getDeploymentContentFile(deploymentHash);
        Path sourcePath = PathUtil.resolveSecurely(contentPath, relativePath);
        try {
            if (Files.exists(contentPath, new LinkOption[0]) && Files.isDirectory(contentPath, new LinkOption[0]) && this.readWrite) {
                Path tmp = PathUtil.createTempDirectory(this.repoRoot.toPath(), CONTENT);
                Path contentDir = tmp.resolve(CONTENT);
                PathUtil.copyRecursively(contentPath, contentDir, true);
                Path targetPath = PathUtil.resolveSecurely(contentDir, relativePath);
                if (!Files.exists(sourcePath, new LinkOption[0])) {
                    throw DeploymentRepositoryLogger.ROOT_LOGGER.archiveNotFound(sourcePath.toString());
                }
                if (Files.isDirectory(sourcePath, new LinkOption[0]) || !PathUtil.isArchive(sourcePath)) {
                    throw DeploymentRepositoryLogger.ROOT_LOGGER.notAnArchive(sourcePath.toString());
                }
                if (Files.exists(targetPath, new LinkOption[0]) && !Files.isDirectory(targetPath, new LinkOption[0])) {
                    PathUtil.deleteRecursively(targetPath);
                }
                PathUtil.unzip(sourcePath, targetPath);
                byte[] sha1Bytes = this.getSha1Bytes(contentDir);
                Path realFile = this.getDeploymentContentFile(sha1Bytes, true);
                if (this.hasContent(sha1Bytes)) {
                    try {
                        PathUtil.deleteRecursively(tmp);
                    }
                    catch (IOException ioex) {
                        DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(ioex, tmp.toAbsolutePath().toString());
                        tmp.toFile().deleteOnExit();
                    }
                    DeploymentRepositoryLogger.ROOT_LOGGER.debugf("Content was already present in repository at location %s", realFile.toAbsolutePath().toString());
                } else {
                    this.moveTempToPermanent(contentDir, realFile);
                    PathUtil.deleteRecursively(tmp);
                    DeploymentRepositoryLogger.ROOT_LOGGER.contentAdded(realFile.toAbsolutePath().toString());
                }
                return sha1Bytes;
            }
            throw DeploymentRepositoryLogger.ROOT_LOGGER.errorExplodingContent(null, sourcePath.toString());
        }
        catch (IOException ioex) {
            DeploymentRepositoryLogger.ROOT_LOGGER.warn(ioex);
            throw DeploymentRepositoryLogger.ROOT_LOGGER.errorExplodingContent(ioex, sourcePath.toString());
        }
    }

    @Override
    public void copyExplodedContent(byte[] deploymentHash, Path target) throws ExplodedContentException {
        Path contentPath = this.getDeploymentContentFile(deploymentHash);
        try {
            if (Files.exists(contentPath, new LinkOption[0]) && Files.isDirectory(contentPath, new LinkOption[0])) {
                PathUtil.copyRecursively(contentPath, target, false);
            }
        }
        catch (IOException ioex) {
            DeploymentRepositoryLogger.ROOT_LOGGER.warn(ioex);
            throw DeploymentRepositoryLogger.ROOT_LOGGER.errorCopyingDeployment(ioex, target.toString());
        }
    }

    @Override
    public void copyExplodedContentFiles(byte[] deploymentHash, List<String> relativePaths, Path target) throws ExplodedContentException {
        Path contentPath = this.getDeploymentContentFile(deploymentHash);
        try {
            if (Files.exists(contentPath, new LinkOption[0]) && Files.isDirectory(contentPath, new LinkOption[0])) {
                for (String relativePath : relativePaths) {
                    PathUtil.copyRecursively(PathUtil.resolveSecurely(contentPath, relativePath), PathUtil.resolveSecurely(target, relativePath), true);
                }
            }
        }
        catch (IOException ioex) {
            DeploymentRepositoryLogger.ROOT_LOGGER.warn(ioex);
            throw DeploymentRepositoryLogger.ROOT_LOGGER.errorCopyingDeployment(ioex, target.toString());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean lock(byte[] hash) throws InterruptedException {
        String hashHex = HashUtil.bytesToHexString(hash);
        Map<String, ReentrantLock> map = this.lockedContents;
        synchronized (map) {
            if (!this.lockedContents.containsKey(hashHex)) {
                this.lockedContents.put(hashHex, new ReentrantLock());
            }
            return this.lockedContents.get(hashHex).tryLock(this.lockTimeout, TimeUnit.MILLISECONDS);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void unlock(byte[] hash) {
        String hashHex = HashUtil.bytesToHexString(hash);
        Map<String, ReentrantLock> map = this.lockedContents;
        synchronized (map) {
            ReentrantLock lock;
            if (this.lockedContents.containsKey(hashHex) && (lock = this.lockedContents.get(hashHex)).isHeldByCurrentThread()) {
                lock.unlock();
                if (!Files.exists(this.getDeploymentContentFile(hash), new LinkOption[0])) {
                    this.lockedContents.remove(hashHex);
                }
            }
        }
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public TypedInputStream readContent(byte[] deploymentHash, String path) throws ExplodedContentException {
        TemporaryFileInputStream temporaryFileInputStream;
        Path tmpDir = null;
        try {
            if (!this.lock(deploymentHash)) {
                throw DeploymentRepositoryLogger.ROOT_LOGGER.errorLockingDeployment();
            }
            Path src = PathUtil.resolveSecurely(this.getDeploymentContentFile(deploymentHash), path);
            tmpDir = Files.createTempDirectory(this.tmpRoot.toPath(), HashUtil.bytesToHexString(deploymentHash), new FileAttribute[0]);
            Path file = PathUtil.readFile(src, tmpDir);
            Path tmp = Files.createTempFile(this.tmpRoot.toPath(), CONTENT, PathUtil.getFileExtension(src), new FileAttribute[0]);
            Files.copy(file, tmp, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
            temporaryFileInputStream = new TemporaryFileInputStream(tmp);
            this.unlock(deploymentHash);
        }
        catch (InterruptedException ex) {
            try {
                Thread.currentThread().interrupt();
                throw new RuntimeException(ex);
                catch (IOException ex2) {
                    DeploymentRepositoryLogger.ROOT_LOGGER.warn(ex2);
                    throw DeploymentRepositoryLogger.ROOT_LOGGER.errorAccessingDeployment(ex2);
                }
            }
            catch (Throwable throwable) {
                this.unlock(deploymentHash);
                PathUtil.deleteSilentlyRecursively(tmpDir);
                throw throwable;
            }
        }
        PathUtil.deleteSilentlyRecursively(tmpDir);
        return temporaryFileInputStream;
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public List<ContentRepositoryElement> listContent(byte[] deploymentHash, String path, ContentFilter filter) throws ExplodedContentException {
        List<ContentRepositoryElement> list;
        block7: {
            Path tmpDir = null;
            try {
                if (!this.lock(deploymentHash)) {
                    throw DeploymentRepositoryLogger.ROOT_LOGGER.errorLockingDeployment();
                }
                tmpDir = Files.createTempDirectory(this.tmpRoot.toPath(), HashUtil.bytesToHexString(deploymentHash), new FileAttribute[0]);
                Path rootPath = PathUtil.resolveSecurely(this.getDeploymentContentFile(deploymentHash), path);
                list = PathUtil.listFiles(rootPath, tmpDir, filter);
                this.unlock(deploymentHash);
                if (tmpDir == null) break block7;
            }
            catch (InterruptedException ex) {
                try {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException(ex);
                    catch (IOException ex2) {
                        DeploymentRepositoryLogger.ROOT_LOGGER.warn(ex2);
                        throw DeploymentRepositoryLogger.ROOT_LOGGER.errorAccessingDeployment(ex2);
                    }
                }
                catch (Throwable throwable) {
                    this.unlock(deploymentHash);
                    if (tmpDir != null) {
                        PathUtil.deleteSilentlyRecursively(tmpDir);
                    }
                    throw throwable;
                }
            }
            PathUtil.deleteSilentlyRecursively(tmpDir);
        }
        return list;
    }

    private void deleteFileWithEmptyAncestorDirectories(Path path) throws IOException {
        if (!this.repoRoot.toPath().equals(path)) {
            Files.deleteIfExists(path);
            Path parent = path.getParent();
            try (Stream<Path> files = Files.list(parent);){
                if (Files.isDirectory(parent, new LinkOption[0]) && files.findAny().isEmpty()) {
                    this.deleteFileWithEmptyAncestorDirectories(parent);
                }
            }
        }
    }

    @Override
    public byte[] addContentToExploded(byte[] deploymentHash, List<ExplodedContent> addFiles, boolean overwrite) throws ExplodedContentException {
        Path contentPath = this.getDeploymentContentFile(deploymentHash);
        try {
            if (Files.exists(contentPath, new LinkOption[0]) && Files.isDirectory(contentPath, new LinkOption[0]) && this.readWrite) {
                Path tmp = PathUtil.createTempDirectory(this.repoRoot.toPath(), CONTENT);
                Path contentDir = tmp.resolve(CONTENT);
                PathUtil.copyRecursively(contentPath, contentDir, overwrite);
                for (ExplodedContent newContent : addFiles) {
                    Path targetFile = PathUtil.resolveSecurely(contentDir, newContent.getRelativePath());
                    if (!Files.exists(targetFile, new LinkOption[0])) {
                        Files.createDirectories(targetFile.getParent(), new FileAttribute[0]);
                    }
                    InputStream in = newContent.getContent();
                    try {
                        if (in == null) {
                            Files.createDirectory(targetFile, new FileAttribute[0]);
                            continue;
                        }
                        if (overwrite) {
                            Files.copy(in, targetFile, StandardCopyOption.REPLACE_EXISTING);
                            continue;
                        }
                        Files.copy(in, targetFile, new CopyOption[0]);
                    }
                    finally {
                        if (in == null) continue;
                        in.close();
                    }
                }
                byte[] sha1Bytes = this.getSha1Bytes(contentDir);
                Path realFile = this.getDeploymentContentFile(sha1Bytes, true);
                if (this.hasContent(sha1Bytes)) {
                    try {
                        PathUtil.deleteRecursively(tmp);
                    }
                    catch (IOException ioex) {
                        DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(ioex, tmp.toAbsolutePath().toString());
                        tmp.toFile().deleteOnExit();
                    }
                    DeploymentRepositoryLogger.ROOT_LOGGER.debugf("Content was already present in repository at location %s", realFile.toAbsolutePath().toString());
                } else {
                    this.moveTempToPermanent(contentDir, realFile);
                    PathUtil.deleteRecursively(tmp);
                    DeploymentRepositoryLogger.ROOT_LOGGER.contentAdded(realFile.toAbsolutePath().toString());
                }
                return sha1Bytes;
            }
            return deploymentHash;
        }
        catch (IOException ex) {
            DeploymentRepositoryLogger.ROOT_LOGGER.warn(ex);
            throw DeploymentRepositoryLogger.ROOT_LOGGER.errorUpdatingDeployment(ex);
        }
    }

    @Override
    public byte[] removeContentFromExploded(byte[] deploymentHash, List<String> paths) throws ExplodedContentException {
        Path contentPath = this.getDeploymentContentFile(deploymentHash);
        try {
            if (Files.exists(contentPath, new LinkOption[0]) && Files.isDirectory(contentPath, new LinkOption[0]) && this.readWrite) {
                Path tmp = PathUtil.createTempDirectory(this.repoRoot.toPath(), CONTENT);
                Path contentDir = tmp.resolve(CONTENT).toAbsolutePath();
                PathUtil.copyRecursively(contentPath, contentDir, false);
                for (String path : paths) {
                    Path targetFile = PathUtil.resolveSecurely(contentDir, path);
                    this.deleteFileWithEmptyAncestorDirectories(targetFile);
                }
                byte[] sha1Bytes = this.getSha1Bytes(contentDir);
                Path realFile = this.getDeploymentContentFile(sha1Bytes, true);
                if (this.hasContent(sha1Bytes)) {
                    try {
                        PathUtil.deleteRecursively(tmp);
                    }
                    catch (IOException ioex) {
                        DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteTempFile(ioex, tmp.toAbsolutePath().toString());
                        tmp.toFile().deleteOnExit();
                    }
                    DeploymentRepositoryLogger.ROOT_LOGGER.debugf("Content was already present in repository at location %s", realFile.toAbsolutePath().toString());
                } else {
                    this.moveTempToPermanent(contentDir, realFile);
                    PathUtil.deleteRecursively(tmp);
                    DeploymentRepositoryLogger.ROOT_LOGGER.contentAdded(realFile.toAbsolutePath().toString());
                }
                return sha1Bytes;
            }
            return deploymentHash;
        }
        catch (IOException ex) {
            DeploymentRepositoryLogger.ROOT_LOGGER.warn(ex);
            throw DeploymentRepositoryLogger.ROOT_LOGGER.errorUpdatingDeployment(ex);
        }
    }

    private byte[] getSha1Bytes(Path path) throws IOException {
        try (MessageDigestHandle handle = new MessageDigestHandle();){
            byte[] byArray = HashUtil.hashPath(handle.getMessageDigest(), path);
            return byArray;
        }
    }

    private static MessageDigest createMessageDigest() {
        try {
            return MessageDigest.getInstance("SHA-1");
        }
        catch (NoSuchAlgorithmException e) {
            throw DeploymentRepositoryLogger.ROOT_LOGGER.cannotObtainSha1(e, MessageDigest.class.getSimpleName());
        }
    }

    private class MessageDigestHandle
    implements AutoCloseable {
        private final MessageDigest digest;
        private final boolean shared;

        private MessageDigestHandle() {
            MessageDigest md = ContentRepositoryImpl.this.messageDigestRef.getAndSet(null);
            this.shared = md != null;
            this.digest = this.shared ? md : ContentRepositoryImpl.createMessageDigest();
        }

        private MessageDigest getMessageDigest() {
            return this.digest;
        }

        @Override
        public void close() {
            if (this.shared) {
                this.digest.reset();
                ContentRepositoryImpl.this.messageDigestRef.set(this.digest);
            }
        }
    }
}

