/*
 * Decompiled with CFR 0.152.
 */
package com.redhat.microprofile.services;

import com.redhat.microprofile.commons.MicroProfileProjectInfo;
import com.redhat.microprofile.commons.metadata.ConverterKind;
import com.redhat.microprofile.commons.metadata.ItemHint;
import com.redhat.microprofile.commons.metadata.ItemMetadata;
import com.redhat.microprofile.ls.commons.BadLocationException;
import com.redhat.microprofile.ls.commons.SnippetsBuilder;
import com.redhat.microprofile.ls.commons.TextDocument;
import com.redhat.microprofile.ls.commons.snippets.TextDocumentSnippetRegistry;
import com.redhat.microprofile.model.Node;
import com.redhat.microprofile.model.PropertiesModel;
import com.redhat.microprofile.model.Property;
import com.redhat.microprofile.model.PropertyKey;
import com.redhat.microprofile.model.values.ValuesRulesManager;
import com.redhat.microprofile.services.QuarkusModel;
import com.redhat.microprofile.settings.MicroProfileCompletionSettings;
import com.redhat.microprofile.settings.MicroProfileFormattingSettings;
import com.redhat.microprofile.snippets.LanguageId;
import com.redhat.microprofile.snippets.SnippetContextForProperties;
import com.redhat.microprofile.utils.DocumentationUtils;
import com.redhat.microprofile.utils.MicroProfilePropertiesUtils;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.eclipse.lsp4j.CompletionItem;
import org.eclipse.lsp4j.CompletionItemKind;
import org.eclipse.lsp4j.CompletionList;
import org.eclipse.lsp4j.InsertTextFormat;
import org.eclipse.lsp4j.Position;
import org.eclipse.lsp4j.Range;
import org.eclipse.lsp4j.TextEdit;
import org.eclipse.lsp4j.jsonrpc.CancelChecker;

class MicroProfileCompletions {
    private static final Logger LOGGER = Logger.getLogger(MicroProfileCompletions.class.getName());
    private TextDocumentSnippetRegistry snippetRegistry;

    MicroProfileCompletions() {
    }

    public CompletionList doComplete(PropertiesModel document, Position position, MicroProfileProjectInfo projectInfo, ValuesRulesManager valuesRulesManager, MicroProfileCompletionSettings completionSettings, MicroProfileFormattingSettings formattingSettings, CancelChecker cancelChecker) {
        CompletionList list = new CompletionList();
        int offset = -1;
        Node node = null;
        try {
            offset = document.offsetAt(position);
            node = document.findNodeAt(offset);
        }
        catch (BadLocationException e) {
            LOGGER.log(Level.SEVERE, "In MicroProfileCompletions, position error", e);
            return list;
        }
        if (node == null) {
            return list;
        }
        switch (node.getNodeType()) {
            case COMMENTS: {
                break;
            }
            case ASSIGN: 
            case PROPERTY_VALUE: {
                MicroProfileCompletions.collectPropertyValueSuggestions(node, document, projectInfo, valuesRulesManager, completionSettings, list);
                break;
            }
            default: {
                MicroProfileCompletions.collectPropertyKeySuggestions(offset, node, document, projectInfo, valuesRulesManager, completionSettings, formattingSettings, list);
                MicroProfileCompletions.collectSnippetSuggestions(offset, node, document, projectInfo, completionSettings, this.getSnippetRegistry(), list);
            }
        }
        return list;
    }

    private static void collectPropertyKeySuggestions(int offset, Node node, PropertiesModel model, MicroProfileProjectInfo projectInfo, ValuesRulesManager valuesRulesManager, MicroProfileCompletionSettings completionSettings, MicroProfileFormattingSettings formattingSettings, CompletionList list) {
        boolean snippetsSupported = completionSettings.isCompletionSnippetsSupported();
        boolean markdownSupported = completionSettings.isDocumentationFormatSupported("markdown");
        Range range = null;
        try {
            range = model.getDocument().lineRangeAt(offset);
        }
        catch (BadLocationException e) {
            LOGGER.log(Level.SEVERE, "In MicroProfileCompletion#collectPropertyKeySuggestions, position error", e);
            return;
        }
        String profile = null;
        if (node != null && node.getNodeType() == Node.NodeType.PROPERTY_KEY) {
            PropertyKey key = (PropertyKey)node;
            if (key.isBeforeProfile(offset)) {
                MicroProfileCompletions.collectProfileSuggestions(offset, key, model, markdownSupported, list);
                return;
            }
            profile = key.getProfile();
        }
        Set<String> existingProperties = MicroProfileCompletions.getExistingProperties(model);
        for (ItemMetadata property : projectInfo.getProperties()) {
            if (property == null) continue;
            String propertyName = property.getName();
            if (profile != null) {
                propertyName = "%" + profile + "." + propertyName;
            }
            if (existingProperties.contains(propertyName) && node.getNodeType() == Node.NodeType.PROPERTY_KEY && !((PropertyKey)node).getPropertyNameWithProfile().equals(propertyName)) continue;
            String name = property.getName();
            CompletionItem item = new CompletionItem(name);
            item.setKind(CompletionItemKind.Property);
            String defaultValue = property.getDefaultValue();
            Collection<ItemHint.ValueHint> enums = MicroProfilePropertiesUtils.getEnums(property, projectInfo, model, valuesRulesManager);
            StringBuilder insertText = new StringBuilder();
            if (profile != null) {
                insertText.append('%');
                insertText.append(profile);
                insertText.append('.');
            }
            MicroProfilePropertiesUtils.FormattedPropertyResult formattedProperty = MicroProfileCompletions.getPropertyName(name, snippetsSupported);
            insertText.append(formattedProperty.getPropertyName());
            String filterText = insertText.toString();
            item.setFilterText(filterText);
            if (formattingSettings.isSurroundEqualsWithSpaces()) {
                insertText.append(' ');
            }
            insertText.append('=');
            if (formattingSettings.isSurroundEqualsWithSpaces()) {
                insertText.append(' ');
            }
            if (enums != null && enums.size() > 0) {
                if (snippetsSupported) {
                    SnippetsBuilder.choice(formattedProperty.getParameterCount() + 1, enums.stream().map(valueHint -> valueHint.getPreferredValue(property.getConverterKinds())).collect(Collectors.toList()), insertText);
                } else {
                    String defaultEnumValue = defaultValue != null ? defaultValue : enums.iterator().next().getValue();
                    insertText.append(defaultEnumValue);
                }
            } else if (defaultValue != null) {
                if (snippetsSupported) {
                    SnippetsBuilder.placeholders(0, defaultValue, insertText);
                } else {
                    insertText.append(defaultValue);
                }
            } else if (snippetsSupported) {
                SnippetsBuilder.tabstops(0, insertText);
            }
            TextEdit textEdit = new TextEdit(range, insertText.toString());
            item.setTextEdit(textEdit);
            item.setInsertTextFormat(snippetsSupported ? InsertTextFormat.Snippet : InsertTextFormat.PlainText);
            item.setDocumentation(DocumentationUtils.getDocumentation(property, profile, markdownSupported));
            list.getItems().add(item);
        }
    }

    private static void collectProfileSuggestions(int offset, PropertyKey key, PropertiesModel model, boolean markdownSupported, CompletionList list) {
        Range range = null;
        Position currPosition = null;
        boolean addPeriod = false;
        String line = null;
        TextDocument textDocument = model.getDocument();
        try {
            range = textDocument.lineRangeAt(offset);
            currPosition = textDocument.positionAt(offset);
            line = textDocument.lineText(currPosition.getLine());
            addPeriod = currPosition.getCharacter() < line.length() && line.charAt(currPosition.getCharacter()) != '.';
        }
        catch (BadLocationException e) {
            LOGGER.log(Level.SEVERE, "In MicroProfileCompletion#collectPropertyKeySuggestions, position error", e);
            return;
        }
        Set profiles = model.getChildren().stream().filter(n -> n.getNodeType() == Node.NodeType.PROPERTY).map(n -> {
            Property property = (Property)n;
            return property.getProfile();
        }).filter(Objects::nonNull).filter(MicroProfileCompletions.not(String::isEmpty)).distinct().collect(Collectors.toSet());
        profiles.addAll(QuarkusModel.getDefaultProfileNames());
        for (String p : profiles) {
            if (p.equals(key.getProfile())) continue;
            CompletionItem item = new CompletionItem(p);
            item.setKind(CompletionItemKind.Struct);
            String insertText = "%" + p + (addPeriod ? "." : "");
            range.setEnd(currPosition);
            TextEdit textEdit = new TextEdit(range, insertText);
            item.setTextEdit(textEdit);
            item.setInsertTextFormat(InsertTextFormat.PlainText);
            item.setFilterText(insertText);
            MicroProfileCompletions.addDocumentationIfDefaultProfile(item, markdownSupported);
            list.getItems().add(item);
        }
    }

    private static void addDocumentationIfDefaultProfile(CompletionItem item, boolean markdown) {
        for (ItemHint.ValueHint profile : QuarkusModel.DEFAULT_PROFILES.getValues()) {
            if (!profile.getValue().equals(item.getLabel())) continue;
            item.setDocumentation(DocumentationUtils.getDocumentation(profile, markdown));
            break;
        }
    }

    private static <T> Predicate<T> not(Predicate<T> t) {
        return t.negate();
    }

    private static Set<String> getExistingProperties(PropertiesModel model) {
        HashSet<String> set = new HashSet<String>();
        for (Node child : model.getChildren()) {
            String name;
            if (child.getNodeType() != Node.NodeType.PROPERTY || (name = ((Property)child).getPropertyNameWithProfile()) == null || name.isEmpty()) continue;
            set.add(name);
        }
        return set;
    }

    private static MicroProfilePropertiesUtils.FormattedPropertyResult getPropertyName(String propertyName, boolean snippetsSupported) {
        if (!snippetsSupported) {
            return new MicroProfilePropertiesUtils.FormattedPropertyResult(propertyName, 0);
        }
        return MicroProfilePropertiesUtils.formatPropertyForCompletion(propertyName);
    }

    private static void collectPropertyValueSuggestions(Node node, PropertiesModel model, MicroProfileProjectInfo projectInfo, ValuesRulesManager valuesRulesManager, MicroProfileCompletionSettings completionSettings, CompletionList list) {
        Collection<ItemHint.ValueHint> enums;
        Node parent = node.getParent();
        if (parent != null && parent.getNodeType() != Node.NodeType.PROPERTY) {
            return;
        }
        Property property = (Property)parent;
        String propertyName = property.getPropertyName();
        ItemMetadata item = MicroProfilePropertiesUtils.getProperty(propertyName, projectInfo);
        if (item != null && (enums = MicroProfilePropertiesUtils.getEnums(item, projectInfo, model, valuesRulesManager)) != null && !enums.isEmpty()) {
            boolean markdownSupported = completionSettings.isDocumentationFormatSupported("markdown");
            for (ItemHint.ValueHint e : enums) {
                list.getItems().add(MicroProfileCompletions.getValueCompletionItem(e, item.getConverterKinds(), node, model, markdownSupported));
            }
        }
    }

    private static CompletionItem getValueCompletionItem(ItemHint.ValueHint item, List<ConverterKind> converterKinds, Node node, PropertiesModel model, boolean markdownSupported) {
        String value = item.getPreferredValue(converterKinds);
        CompletionItem completionItem = new CompletionItem(value);
        completionItem.setKind(CompletionItemKind.Value);
        Range range = null;
        try {
            TextDocument doc = model.getDocument();
            int startOffset = node.getNodeType() == Node.NodeType.ASSIGN ? node.getEnd() : node.getStart();
            range = doc.lineRangeAt(startOffset);
            range.setStart(doc.positionAt(startOffset));
        }
        catch (BadLocationException e) {
            LOGGER.log(Level.SEVERE, "In MicroProfileCompletion#getEnumCompletionItem, position error", e);
        }
        TextEdit textEdit = new TextEdit(range, value);
        completionItem.setTextEdit(textEdit);
        completionItem.setDocumentation(DocumentationUtils.getDocumentation(item, markdownSupported));
        return completionItem;
    }

    private static void collectSnippetSuggestions(int completionOffset, Node node, PropertiesModel document, MicroProfileProjectInfo projectInfo, MicroProfileCompletionSettings completionSettings, TextDocumentSnippetRegistry snippetRegistry, CompletionList list) {
        boolean markdownSupported = completionSettings.isDocumentationFormatSupported("markdown");
        snippetRegistry.getCompletionItems(document.getDocument(), completionOffset, markdownSupported, context -> {
            if (context instanceof SnippetContextForProperties) {
                SnippetContextForProperties contextProperties = (SnippetContextForProperties)context;
                return contextProperties.isMatch(projectInfo);
            }
            return false;
        }).forEach(item -> list.getItems().add(item));
    }

    private TextDocumentSnippetRegistry getSnippetRegistry() {
        if (this.snippetRegistry == null) {
            this.snippetRegistry = new TextDocumentSnippetRegistry(LanguageId.properties.name()){

                @Override
                protected String getExpr(TextDocument document, int completionOffset) {
                    String expr = super.getExpr(document, completionOffset);
                    if (expr.length() > 0 && expr.charAt(0) == '%') {
                        return null;
                    }
                    return expr;
                }
            };
        }
        return this.snippetRegistry;
    }
}

