/*
 * Decompiled with CFR 0.152.
 */
package com.redhat.microprofile.services;

import com.redhat.microprofile.commons.MicroProfileProjectInfo;
import com.redhat.microprofile.commons.metadata.ConfigurationMetadata;
import com.redhat.microprofile.commons.metadata.ItemMetadata;
import com.redhat.microprofile.model.Node;
import com.redhat.microprofile.model.PropertiesModel;
import com.redhat.microprofile.model.Property;
import com.redhat.microprofile.model.values.ValuesRulesManager;
import com.redhat.microprofile.services.ValidationType;
import com.redhat.microprofile.settings.MicroProfileValidationSettings;
import com.redhat.microprofile.utils.MicroProfilePropertiesUtils;
import com.redhat.microprofile.utils.PositionUtils;
import com.redhat.microprofile.utils.StringUtils;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.eclipse.lsp4j.Diagnostic;
import org.eclipse.lsp4j.DiagnosticSeverity;
import org.eclipse.lsp4j.Range;
import org.eclipse.lsp4j.jsonrpc.CancelChecker;

class MicroProfileValidator {
    private static final String MICROPROFILE_DIAGNOSTIC_SOURCE = "microprofile";
    private final MicroProfileProjectInfo projectInfo;
    private final ValuesRulesManager valuesRulesManager;
    private final List<Diagnostic> diagnostics;
    private final MicroProfileValidationSettings validationSettings;
    private final Map<String, List<Property>> existingProperties;

    public MicroProfileValidator(MicroProfileProjectInfo projectInfo, ValuesRulesManager valuesRulesManager, List<Diagnostic> diagnostics, MicroProfileValidationSettings validationSettings) {
        this.projectInfo = projectInfo;
        this.valuesRulesManager = valuesRulesManager;
        this.diagnostics = diagnostics;
        this.validationSettings = validationSettings;
        this.existingProperties = new HashMap<String, List<Property>>();
    }

    public void validate(PropertiesModel document, CancelChecker cancelChecker) {
        List<Node> nodes = document.getChildren();
        for (Node node : nodes) {
            if (cancelChecker != null) {
                cancelChecker.checkCanceled();
            }
            if (node.getNodeType() != Node.NodeType.PROPERTY) continue;
            this.validateProperty((Property)node);
        }
        this.addDiagnosticsForDuplicates();
        this.addDiagnosticsForMissingRequired(document);
    }

    private void validateProperty(Property property) {
        String propertyName;
        String propertyNameWithProfile = property.getPropertyNameWithProfile();
        if (propertyNameWithProfile != null && !propertyNameWithProfile.isEmpty()) {
            this.validateSyntaxProperty(propertyNameWithProfile, property);
            this.validateDuplicateProperty(propertyNameWithProfile, property);
        }
        if ((propertyName = property.getPropertyName()) != null && !propertyName.isEmpty()) {
            ItemMetadata metadata = MicroProfilePropertiesUtils.getProperty(propertyName, this.projectInfo);
            if (metadata == null) {
                this.validateUnknownProperty(propertyNameWithProfile, property);
            } else {
                this.validatePropertyValue(propertyNameWithProfile, metadata, property);
            }
        }
    }

    private void validateSyntaxProperty(String propertyName, Property property) {
        DiagnosticSeverity severity = this.validationSettings.getSyntax().getDiagnosticSeverity(propertyName);
        if (severity == null) {
            return;
        }
        if (property.getDelimiterAssign() == null) {
            this.addDiagnostic("Missing equals sign after '" + propertyName + "'", property.getKey(), severity, ValidationType.syntax.name());
        }
    }

    private void validateDuplicateProperty(String propertyName, Property property) {
        DiagnosticSeverity severity = this.validationSettings.getDuplicate().getDiagnosticSeverity(propertyName);
        if (severity == null) {
            return;
        }
        if (!this.existingProperties.containsKey(propertyName)) {
            this.existingProperties.put(propertyName, new ArrayList());
        }
        this.existingProperties.get(propertyName).add(property);
    }

    private void validateUnknownProperty(String propertyName, Property property) {
        DiagnosticSeverity severity = this.validationSettings.getUnknown().getDiagnosticSeverity(propertyName);
        if (severity == null) {
            return;
        }
        this.addDiagnostic("Unknown property '" + propertyName + "'", property.getKey(), severity, ValidationType.unknown.name());
    }

    private void validatePropertyValue(String propertyName, ItemMetadata metadata, Property property) {
        if (property.getValue() == null) {
            return;
        }
        DiagnosticSeverity severity = this.validationSettings.getValue().getDiagnosticSeverity(propertyName);
        if (severity == null) {
            return;
        }
        String value = property.getPropertyValue();
        if (value == null || value.isEmpty()) {
            return;
        }
        String errorMessage = this.getErrorIfInvalidEnum(metadata, this.projectInfo, property.getOwnerModel(), value);
        if (errorMessage == null) {
            errorMessage = MicroProfileValidator.getErrorIfValueTypeMismatch(metadata, value);
        }
        if (errorMessage != null) {
            this.addDiagnostic(errorMessage, property.getValue(), severity, ValidationType.value.name());
        }
    }

    private String getErrorIfInvalidEnum(ItemMetadata metadata, ConfigurationMetadata configuration, PropertiesModel model, String value) {
        if (!configuration.isValidEnum(metadata, value) || this.valuesRulesManager != null && !this.valuesRulesManager.isValidEnum(metadata, model, value)) {
            return "Invalid enum value: '" + value + "' is invalid for type " + metadata.getType();
        }
        return null;
    }

    private static String getErrorIfValueTypeMismatch(ItemMetadata metadata, String value) {
        if (MicroProfileValidator.isBuildtimePlaceholder(value)) {
            return null;
        }
        if (metadata.isRegexType()) {
            try {
                Pattern.compile(value);
                return null;
            }
            catch (PatternSyntaxException e) {
                return e.getMessage() + System.lineSeparator();
            }
        }
        if (metadata.isIntegerType() && !MicroProfileValidator.isIntegerString(value) || metadata.isFloatType() && !MicroProfileValidator.isFloatString(value) || metadata.isBooleanType() && !MicroProfileValidator.isBooleanString(value) || metadata.isDoubleType() && !MicroProfileValidator.isDoubleString(value) || metadata.isLongType() && !MicroProfileValidator.isLongString(value) || metadata.isShortType() && !MicroProfileValidator.isShortString(value) || metadata.isBigDecimalType() && !MicroProfileValidator.isBigDecimalString(value) || metadata.isBigIntegerType() && !MicroProfileValidator.isBigIntegerString(value)) {
            return "Type mismatch: " + metadata.getType() + " expected";
        }
        return null;
    }

    private static boolean isBooleanString(String str) {
        return "true".equals(str) || "false".equals(str);
    }

    private static boolean isIntegerString(String str) {
        if (!StringUtils.hasText(str)) {
            return false;
        }
        try {
            Integer.parseInt(str);
            return true;
        }
        catch (NumberFormatException e) {
            return false;
        }
    }

    private static boolean isFloatString(String str) {
        if (!StringUtils.hasText(str)) {
            return false;
        }
        try {
            Float.parseFloat(str);
            return true;
        }
        catch (NumberFormatException e) {
            return false;
        }
    }

    private static boolean isLongString(String str) {
        if (!StringUtils.hasText(str)) {
            return false;
        }
        try {
            Long.parseLong(str);
            return true;
        }
        catch (NumberFormatException e) {
            return false;
        }
    }

    private static boolean isDoubleString(String str) {
        if (!StringUtils.hasText(str)) {
            return false;
        }
        try {
            Double.parseDouble(str);
            return true;
        }
        catch (NumberFormatException e) {
            return false;
        }
    }

    private static boolean isShortString(String str) {
        if (!StringUtils.hasText(str)) {
            return false;
        }
        try {
            Short.parseShort(str);
            return true;
        }
        catch (NumberFormatException e) {
            return false;
        }
    }

    private static boolean isBigDecimalString(String str) {
        if (!StringUtils.hasText(str)) {
            return false;
        }
        try {
            new BigDecimal(str);
            return true;
        }
        catch (NumberFormatException e) {
            return false;
        }
    }

    private static boolean isBigIntegerString(String str) {
        if (!StringUtils.hasText(str)) {
            return false;
        }
        try {
            new BigInteger(str);
            return true;
        }
        catch (NumberFormatException e) {
            return false;
        }
    }

    private static boolean isBuildtimePlaceholder(String str) {
        return str.startsWith("${") && str.endsWith("}");
    }

    private void addDiagnosticsForDuplicates() {
        this.existingProperties.forEach((propertyName, propertyList) -> {
            if (propertyList.size() <= 1) {
                return;
            }
            DiagnosticSeverity severity = this.validationSettings.getDuplicate().getDiagnosticSeverity((String)propertyName);
            for (Property property : propertyList) {
                this.addDiagnostic("Duplicate property '" + propertyName + "'", property.getKey(), severity, ValidationType.duplicate.name());
            }
        });
    }

    private void addDiagnosticsForMissingRequired(PropertiesModel document) {
        for (ItemMetadata property : this.projectInfo.getProperties()) {
            String propertyName = property.getName();
            DiagnosticSeverity severity = this.validationSettings.getRequired().getDiagnosticSeverity(propertyName);
            if (severity == null || !property.isRequired()) continue;
            if (!this.existingProperties.containsKey(propertyName)) {
                this.addDiagnostic("Missing required property '" + propertyName + "'", document, severity, ValidationType.required.name());
                continue;
            }
            this.addDiagnosticsForRequiredIfNoValue(propertyName, severity);
        }
    }

    private void addDiagnosticsForRequiredIfNoValue(String propertyName, DiagnosticSeverity severity) {
        List<Property> propertyList = this.existingProperties.get(propertyName);
        for (Property property : propertyList) {
            if (property.getValue() == null || property.getValue().getValue().isEmpty()) continue;
            return;
        }
        for (Property property : propertyList) {
            this.addDiagnostic("Missing required property value for '" + propertyName + "'", property, severity, ValidationType.requiredValue.name());
        }
    }

    private void addDiagnostic(String message, Node node, DiagnosticSeverity severity, String code) {
        Range range = PositionUtils.createRange(node);
        this.diagnostics.add(new Diagnostic(range, message, severity, MICROPROFILE_DIAGNOSTIC_SOURCE, code));
    }

    public MicroProfileValidationSettings getValidationSettings() {
        return this.validationSettings;
    }
}

