/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * Copyright (c) 2013, Red Hat, Inc. and/or its affiliates or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat, Inc.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.hibernate.search.metadata;

import java.util.Set;

import org.hibernate.search.engine.BoostStrategy;

/**
 * Top level descriptor of the metadata API. Giving access to the indexing information for a single type.
 *
 * @author Hardy Ferentschik
 */
public interface IndexedTypeDescriptor extends FieldContributor {

	/**
	 * @return the type for which this descriptor provides meta information
	 */
	Class<?> getType();

	/**
	 * @return {@code true} if the type for this descriptor is indexed, {@code false} otherwise
	 */
	boolean isIndexed();

	/**
	 * @return {@code true} is this index is sharded, {@code false} otherwise
	 */
	boolean isSharded();

	/**
	 * Returns a set of {@code IndexDescriptor} instances describing Lucene index information, in particular the index name.
	 * The index name can also be used to retrieve the actual {@code IndexManager} for this index via
	 * {@link org.hibernate.search.engine.spi.SearchFactoryImplementor#getIndexManagerHolder()#getIndexManager(String)}.
	 *
	 * For non sharded indexes there will be only a single {@code IndexDescriptor}. The empty set is returned for an unindexed type
	 *
	 * @return a set of {@code IndexDescriptor} instances describing Lucene index information
	 */
	Set<IndexDescriptor> getIndexDescriptors();

	/**
	 * @return the class boost value, 1 being the default.
	 */
	float getStaticBoost();

	/**
	 * @return Dynamic boost strategy. There will always be a boost strategy, but the default strategy will apply a
	 *         boost of 1.0.
	 */
	BoostStrategy getDynamicBoost();

	/**
	 * @return the set of indexed properties in form of {@code PropertyDescriptor}s
	 */
	Set<PropertyDescriptor> getIndexedProperties();

	/**
	 * Retrieves the property descriptor for a given property name.
	 *
	 * @param propertyName the property name for which to return descriptor, cannot be {@code null}
	 *
	 * @return the property descriptor for the specified property name. {@code null} is returned in case a property with
	 *         the specified name does not exist
	 *
	 * @throws IllegalArgumentException in case {@code propertyName} is {@code null}
	 */
	PropertyDescriptor getProperty(String propertyName);

	/**
	 * Retrieves the field descriptor for a given field name.
	 *
	 * @param fieldName the field name for which to return descriptor, cannot be {@code null}
	 *
	 * @return the field descriptor for the specified field name. {@code null} is returned in case a field with the specified name does not exist
	 *
	 * @throws IllegalArgumentException in case {@code fieldName} is {@code null}
	 */
	FieldDescriptor getIndexedField(String fieldName);

	/**
	 * Returns the set of index descriptors for the indexed field generated by the property with the given name.
	 *
	 * @param propertyName the property name, cannot be {@code null}
	 *
	 * @return the set of index descriptors for the indexed field generated by the property with the given name. The empty set
	 *         is returned in case the property does not exist or does not produce any indexed fields.
	 *
	 * @throws IllegalArgumentException in case {@code propertyName} is {@code null}
	 */
	Set<FieldDescriptor> getFieldsForProperty(String propertyName);
}
